﻿package 
{
	
	/**
	 * ...
	 * @author Daniel Bienvenu
	 */
	public class SoundChip 
	{
		private static var ATTENUATION_MAX:int = 15;
		//private static var TIMEUNIT_NTSC:Number = 1.0 / 59.94;
		//private static var TIMEUNIT_PAL:Number = 1.0 / 50.0;
		private static var CYCLESINTIME:Number = 3579545.0 / 16.0;
		
		private var generaltime:Number;
		
		private var interruption:Number;
		private var interruption_counter:Number;
		
		private var flipflop:Array = new Array(4);
		private var counter:Array = new Array(4);
		private var period:Array = new Array(4);
		private var attenuation:Array = new Array(4);
		private var noise_freq3:Boolean;
		private var noise_white:Boolean;
		private var noise_oldcode:int;
		
		private var ShiftRegister:int;
		private static var ShiftRegister_PRESET:int = 0x4000;
		
		private var Engine:SoundEngine;
		
		private static var amplification:Array = [
			0.25,
			0.1985820586810703755164795707091,
			0.15773933612004831235859003415559,
			0.12529680840681807125038854672124,
			0.099526792638374312692563076271938,
			0.079056941504209483299972338610818,
			0.062797160787739502777125801694983,
			0.049881557874221990033811384918488,
			0.039622329811527837130052534334788,
			0.031473135294854180260598852659895,
			0.025,
			0.01985820586810703755164795707091,
			0.015773933612004831235859003415559,
			0.012529680840681807125038854672124,
			0.0099526792638374312692563076271938,
			0.0,
		];
		
		private static var NoiseFreqs:Array = [32, 64, 128, 0];
		
		public function SoundChip(engine:SoundEngine = null, freq:Number = 59.94) { // default NTSC freq
			
			setBIOS(engine);
			
			// Set INTERRUPTION FRENQUENCY (normally between PAL and NTSC)
			setSpeedHz(freq);
			interruption_counter = 0.0;
			
			reset_channels();
			
			// TEST
			//noise_coleco(6, 0);
			/*
			period[1] = 0x1AB; // 1.0 / 130.81; // C3
			period[2] = 0x11D; // 1.0 / 196.00; // G3
			period[3] = 0x0D5; // 1.0 / 261.63; // C4
			attenuation[1] = 6; // volume = 9/15
			attenuation[2] = 4; // volume = 11/15
			attenuation[3] = 2; // volume = 13/15
			*/
		}
		
		public function setSpeedHz(Hz:Number):void {
			interruption = 1.0/Hz;
		}
		
		public function setBIOS(new_engine:SoundEngine):void {
			if (new_engine != null) {
				Engine = new_engine;
				Engine.setSoundChip(this);
			}
		}
		
		private function reset_channels():void {
			// reset channels vars
			for (var i:int; i < 4; i++) {
				attenuation[i] = ATTENUATION_MAX;
				counter[i] = 0; // 0.0;
				flipflop[i] = -1.0;
				period[i] = 0; //0.0;
			}
			ShiftRegister = ShiftRegister_PRESET;
			noise_freq3 = false;
			noise_white = true;
			noise_oldcode = -1;
			
			generaltime = 0.0;
		}
		
		public function tone(channel:int, new_period:int, new_attenuation:int):void {
			attenuation[channel] = new_attenuation;
			period[channel] = new_period;
			if (new_attenuation == ATTENUATION_MAX) {
				counter[channel] = 0; // 0.0;
			}
		}
		
		public function noise_coleco(code:int, new_attenuation:int):void {
			noise_attenuation(new_attenuation);
			if (code != noise_oldcode) {
				noise_oldcode = code;
				noise_code(code);
			}
		}
		
		public function noise_attenuation(new_attenuation:int):void {
			attenuation[0] = new_attenuation;
			if (new_attenuation == ATTENUATION_MAX) {
				counter[0] = 0; // 0.0;
			}
		}
		
		public function noise_code(code:int):void {
			noise_oldcode = code;
			ShiftRegister = ShiftRegister_PRESET;
			noise_white = ((code & 4) == 4);
			code &= 3;
			noise_freq3 = (code == 3);
			noise_period_update();
			counter[0] = 0; // 0.0;
		}
		
		public function noise_period_update():void {
			if (noise_freq3) {
				period[0] = period[3] * 2;
			} else {
				period[0] = NoiseFreqs[noise_oldcode&3];
			}			
		}
		
		/*
		 * tonefreq (Hz) = 3579545 / (32*reg)
		 */
		
		public function update(time:Number):void {
			
			interruption_counter += time;
			if (interruption_counter >= interruption) {
				interruption_counter = interruption_counter - interruption;				
				if (Engine != null) Engine.update_sound();
			}
			
			var nbcycles:int = Math.floor(CYCLESINTIME * (generaltime + time)) - Math.floor(CYCLESINTIME * generaltime);
			generaltime += time;
			
			noise_period_update();
			
			for (var i:int = 4; i >=0 ; i--)	{
				if (attenuation[i] < ATTENUATION_MAX) {
					counter[i] += nbcycles;
					if (counter[i] >= period[i]) {
						if (i == 0) {							
							flipflop[0] = ((ShiftRegister & 1) == 1)? -1.0 : 1.0;
							var feedback:int;
							if (noise_white) {
								feedback = (ShiftRegister ^ ( ShiftRegister >>> 1)) & 1;
							} else {
								feedback = ShiftRegister & 1;
							}
							ShiftRegister = (ShiftRegister >>> 1) | (feedback << 14);							
						} else {
							flipflop[i] = -flipflop[i];
						}
						counter[i] %= period[i];
					}
				} else {
					counter[i] = 0; // 0.0;
				}
			}			
		}
		
		public function output():Number {
			var out:Number = 0.0;
			for (var i:int = 0; i < 4; i++) {
				out += flipflop[i] * amplification[attenuation[i]];
			}
			return out;
		}
		
		public function str_channel_note(channel:int):String {
			var note:int;
			var frequency:Number;
			var letters:Array = ["C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "Bb", "B"];
			
			if (channel > 0) {
				frequency = (CYCLESINTIME / period[channel]) / 2;
			} else {
				if (noise_white) {
					frequency = (CYCLESINTIME / period[channel]) / 2;
				} else {
					frequency = (CYCLESINTIME / period[channel]) / 14;
				}
			}
			note = Math.round(12 * Math.log(frequency / 440.0) / Math.log(2)) + 57; // 57 = A4 = 440.0 Hz
			return letters[note % 12] + String(Math.floor(note / 12));
		}
		
	}
}